using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

namespace EQTrack
{
	/// <summary>
	/// Summary description for TransparentLabel.
	/// </summary>
	public class CustomLabel : System.Windows.Forms.Label
	{
		private bool mAutoSize = false;
		private int mFontSizeAdjust = 0;
		private float mFontSizeMultiplier = 1.0f;
		private FontStyle mFontStyleOverride = FontStyle.Regular;
		private bool mShowTextShadow = false;
		private Color mTextShadowColor = Color.LightGray;
		private Color mForeColorHover = SystemColors.ControlText;
		private bool IsHovering = false;

		/// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		public CustomLabel()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			this.SetStyle(ControlStyles.UserPaint, true);
			this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
			this.SetStyle(ControlStyles.DoubleBuffer, true);
		}

		public override string Text {
			get {
				return base.Text;
			}
			set {
				base.Text = value;
			}
		}

		[
		Category("Appearance"),
		DefaultValue(0)
		]
		public int FontSizeAdjust {
			get {
				return mFontSizeAdjust;
			}
			set {
				if (mFontSizeAdjust != value) {
					mFontSizeAdjust = value;
					AdjustSize();
					this.Refresh();
				}
			}
		}

		[
		Category("Appearance"),
		DefaultValue(1.0f)
		]
		public float FontSizeMultiplier {
			get {
				return mFontSizeMultiplier;
			}
			set {
				if (mFontSizeMultiplier != value) {
					mFontSizeMultiplier = value;
					AdjustSize();
					this.Refresh();
				}
			}
		}

		[
		Category("Appearance"),
		DefaultValue(FontStyle.Regular)
		]
		public FontStyle FontStyleOverride {
			get {
				return mFontStyleOverride;
			}
			set {
				if (mFontStyleOverride != value) {
					mFontStyleOverride = value;
					AdjustSize();
					this.Refresh();
				}
			}
		}

		[
		Category("Appearance"),
		DefaultValue(false)
		]
		public bool ShowTextShadow {
			get {
				return mShowTextShadow;
			}
			set {
				if (mShowTextShadow != value) {
					mShowTextShadow = value;
					this.Refresh();
				}
			}
		}

		[
		Category("Appearance"),
		]
		public Color TextShadowColor {
			get {
				return mTextShadowColor;
			}
			set {
				if (mTextShadowColor != value) {
					mTextShadowColor = value;
					this.Refresh();
				}
			}
		}

		[
		Category("Appearance"),
		]
		public Color ForeColorHover
		{
			get
			{
				return mForeColorHover;
			}
			set
			{
				if (mForeColorHover != value)
				{
					mForeColorHover = value;
					this.Refresh();
				}
			}
		}

		[
		Category("Appearance"),
		DefaultValue(false)
		]
		public new bool AutoSize {
			get {
				return mAutoSize;
			}
			set {
				if (mAutoSize != value) {
					mAutoSize = value;
					AdjustSize();
					this.Refresh();
				}
			}
		}

		private void AdjustSize() {
			if (mAutoSize) {
				Font ThisFont = new Font(this.Font.Name, (this.Font.Size * mFontSizeMultiplier) + mFontSizeAdjust, mFontStyleOverride);
				Size ThisSize = Graphics.FromHwnd(this.Handle).MeasureString(this.Text, ThisFont).ToSize();
				ThisSize.Width = ThisSize.Width + 2;
				this.Size = ThisSize;
			}
		}

		protected override void OnFontChanged(EventArgs e) {
			base.OnFontChanged (e);
			AdjustSize();
		}

		protected override void OnTextChanged(EventArgs e) {
			base.OnTextChanged (e);
			AdjustSize();
		}

		protected override void OnPaint(PaintEventArgs e) {
			Font ThisFont = new Font(this.Font.Name, (this.Font.Size * mFontSizeMultiplier) + mFontSizeAdjust, mFontStyleOverride);
			Size TextSize = e.Graphics.MeasureString(this.Text, ThisFont).ToSize();
			int ThisLeft = 0;
			if (this.TextAlign == ContentAlignment.TopCenter || this.TextAlign == ContentAlignment.MiddleCenter || this.TextAlign == ContentAlignment.BottomCenter)
				ThisLeft = (this.Width / 2) - (TextSize.Width / 2);
			else if (this.TextAlign == ContentAlignment.TopRight || this.TextAlign == ContentAlignment.MiddleRight || this.TextAlign == ContentAlignment.BottomRight)
				ThisLeft = this.Width - TextSize.Width;
			int ThisTop = 0;
			if (this.TextAlign == ContentAlignment.MiddleLeft || this.TextAlign == ContentAlignment.MiddleCenter || this.TextAlign == ContentAlignment.MiddleRight)
				ThisTop = (this.Height / 2) - (TextSize.Height / 2);
			else if (this.TextAlign == ContentAlignment.BottomLeft || this.TextAlign == ContentAlignment.BottomCenter || this.TextAlign == ContentAlignment.BottomRight)
				ThisTop = this.Height - TextSize.Height;
			if (mShowTextShadow)
				e.Graphics.DrawString(this.Text, ThisFont, new SolidBrush(mTextShadowColor), ThisLeft + 1, ThisTop + 1);
			e.Graphics.DrawString(this.Text, ThisFont, new SolidBrush(IsHovering ? this.mForeColorHover : this.ForeColor), ThisLeft, ThisTop);
		}

		protected override void OnMouseEnter(EventArgs e)
		{
			base.OnMouseEnter(e);
			IsHovering = true;
			if (mForeColorHover != ForeColor)
				this.Refresh();
		}

		protected override void OnMouseLeave(EventArgs e)
		{
			base.OnMouseLeave(e);
			IsHovering = false;
			if (mForeColorHover != ForeColor)
				this.Refresh();
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			components = new System.ComponentModel.Container();
		}
		#endregion
	}
}
